<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Models\Cobro;
use App\Models\Pelicula;
use App\Models\Alquiler;
use App\Models\Sala;
use App\Models\VentaDetalle;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        if (!auth()->user()->can('crear peliculas')) {
            return redirect()->route('dashboard.personal');
            // Esta puede ser tu vista personalizada
        }
        $startDate = $request->input('startDate', now()->subDays(7)->format('Y-m-d'));
        $endDate = $request->input('endDate', now()->format('Y-m-d'));

        $totalPeliculas = Pelicula::count();
        $totalAlquileres = Alquiler::count();
        $totalSalas = Sala::count();
        $salasOcupadas = Sala::where('estado', 'ocupado')->count();
        $salasDisponibles = max(0, $totalSalas - $salasOcupadas);
        $alertaSalas = $salasDisponibles < 5;

        $totalCobros = Alquiler::sum('precio') ?? 0;
        $ingresosSemana = Alquiler::whereBetween('created_at', [now()->subDays(7), now()])->sum('precio') ?? 0;
        $ingresosMes = Alquiler::whereMonth('created_at', now()->month)->sum('precio') ?? 0;

        $ingresosPorDia = Alquiler::join('peliculas', 'alquileres.pelicula_id', '=', 'peliculas.id')
            ->selectRaw('DATE(alquileres.created_at) as fecha, SUM(alquileres.precio) as total')
            ->whereBetween('alquileres.created_at', [$startDate, $endDate])
            ->groupBy('fecha')
            ->orderBy('fecha', 'DESC')
            ->get();

        $fechas = $ingresosPorDia->pluck('fecha')->toArray();
        $montos = $ingresosPorDia->pluck('total')->toArray();

        $topPeliculas = DB::table('alquileres')
            ->join('peliculas', 'alquileres.pelicula_id', '=', 'peliculas.id')
            ->select('peliculas.titulo as nombre', DB::raw('COUNT(alquileres.id) as total'))
            ->groupBy('peliculas.titulo')
            ->orderByDesc('total')
            ->limit(5)
            ->get();

        $ultimosCobros = Cobro::join('alquileres', 'cobros.alquiler_id', '=', 'alquileres.id')
            ->join('peliculas', 'alquileres.pelicula_id', '=', 'peliculas.id')
            ->select('cobros.fecha_cobro', 'peliculas.titulo', 'alquileres.precio')
            ->orderByDesc('cobros.fecha_cobro')
            ->limit(3)
            ->get();

        $ingresosDiariosPeliculas = Alquiler::whereDate('created_at', now())->sum('precio');
        $ingresosDiariosProductos = VentaDetalle::whereDate('created_at', now())->sum('subtotal');

        $ventasAlquiler = Alquiler::whereBetween('created_at', [$startDate, $endDate])->sum('precio');
        $ventasProductos = VentaDetalle::whereBetween('created_at', [$startDate, $endDate])->sum('subtotal');

        $ventasTotales = $ventasAlquiler + $ventasProductos;

        $ingresosTotalesHoy = $ingresosDiariosPeliculas + $ingresosDiariosProductos;

        return view('dashboard', compact(
            'totalPeliculas', 'totalAlquileres', 'totalSalas', 'salasDisponibles', 'alertaSalas',
            'totalCobros', 'ingresosSemana', 'ingresosMes', 'fechas', 'montos',
            'topPeliculas', 'ultimosCobros', 'startDate', 'endDate',
            'ventasAlquiler', 'ventasProductos', 'ventasTotales',
            'ingresosDiariosPeliculas', 'ingresosDiariosProductos','ingresosTotalesHoy'
        ));
    }

    public function ingresosDiarios(Request $request)
    {
        $startDate = $request->input('startDate');
        $endDate = $request->input('endDate');

        if (!$startDate || !$endDate) {
            return response()->json(['error' => 'Fechas no válidas'], 400);
        }

        $ingresosPorDia = Alquiler::join('peliculas', 'alquileres.pelicula_id', '=', 'peliculas.id')
            ->selectRaw('DATE(alquileres.created_at) as fecha, SUM(alquileres.precio) as total')
            ->groupBy('fecha')
            ->whereBetween('alquileres.created_at', [$startDate, $endDate])
            ->get();

        if ($ingresosPorDia->isEmpty()) {
            return response()->json([
                'fechas' => [],
                'montos' => []
            ]);
        }

        return response()->json([
            'fechas' => $ingresosPorDia->pluck('fecha')->toArray(),
            'montos' => $ingresosPorDia->pluck('total')->toArray()
        ]);
    }

    public function getUltimosCobros()
    {
        $ultimosCobros = Cobro::join('alquileres', 'cobros.alquiler_id', '=', 'alquileres.id')
            ->join('peliculas', 'alquileres.pelicula_id', '=', 'peliculas.id')
            ->select('cobros.fecha_cobro', 'peliculas.titulo', 'alquileres.precio')
            ->orderByDesc('cobros.fecha_cobro')
            ->limit(3)
            ->get();

        return response()->json($ultimosCobros);
    }
}