<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use App\Models\Producto;
use App\Models\Venta;
use App\Models\Alquiler; // Agregar esta línea
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Barryvdh\DomPDF\Facade\Pdf;
use Rap2hpoutre\FastExcel\FastExcel;

class ProductoController extends Controller
{
    public function index(Request $request)
{
    if (!auth()->user()->can('crear peliculas')) {
        return view('errors.access-denied'); // Mostrar una vista personalizada si no tiene permiso
    }

    $umbral = 5;

    $query = Producto::query();

    if ($request->filled('stock')) {
        if ($request->stock == 'bajo') {
            $query->where('stock', '<', $umbral)->where('stock', '>', 0);
        } elseif ($request->stock == 'agotado') {
            $query->where('stock', 0);
        } elseif ($request->stock == 'normal') {
            $query->where('stock', '>=', $umbral);
        }
    }

    if ($request->filled('search')) {
        $query->where('nombre', 'like', '%' . $request->search . '%');
    }

    $productos = $query->paginate(4)->appends($request->query());
    $lowStockCount = Producto::where('stock', '<', $umbral)->count();

    return view('productos.index', compact('productos', 'lowStockCount'));
}



    public function create()
    {
        if (!auth()->user()->can('crear peliculas')) {
            return view('errors.access-denied'); 
        }
        return view('productos.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'precio' => 'required|numeric|min:0',
            'stock' => 'required|integer|min:0',
            
            'imagen' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // Validación de imagen
        ]);
    
        // Crear un nuevo producto sin la imagen
        $producto = new Producto($request->except('imagen'));
    
        // Verificar si se ha subido una imagen
        if ($request->hasFile('imagen')) {
            $imagenPath = $request->file('imagen')->store('imagenes', 'public');
            $producto->imagen = $imagenPath;
        }
    
        $producto->save();
    
        return redirect()->route('productos.index')->with('success', 'Producto registrado correctamente.');
    }
    

    public function show($id)
    {
        $producto = Producto::findOrFail($id);
        return view('productos.show', compact('producto'));
    }

    public function edit($id)
    {
        $producto = Producto::findOrFail($id);
        return view('productos.edit', compact('producto'));
    }

    public function update(Request $request, $id)
    {
        $producto = Producto::findOrFail($id);
    
        // Validación de campos
        $validatedData = $request->validate([
            'nombre' => 'required|string|max:255',
            'precio' => 'required|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'imagen' => 'nullable|image|mimes:jpeg,png,jpg|max:2048', // 2MB máximo
            'eliminar_imagen' => 'nullable|boolean'
        ]);
    
        // Manejo de la imagen
        if ($request->has('eliminar_imagen') && $request->eliminar_imagen) {
            // Eliminar imagen existente si se marca el checkbox
            if ($producto->imagen) {
                Storage::delete('public/' . $producto->imagen);
                $producto->imagen = null;
            }
        } elseif ($request->hasFile('imagen')) {
            // Eliminar imagen anterior si existe
            if ($producto->imagen) {
                Storage::delete('public/' . $producto->imagen);
            }
            
            // Guardar nueva imagen
            $imagePath = $request->file('imagen')->store('productos', 'public');
            $validatedData['imagen'] = $imagePath;
        }
    
        // Actualizar el producto
        $producto->update($validatedData);
    
        return redirect()->route('productos.index')
            ->with('success', 'Producto actualizado correctamente.');
    }

    public function destroy($id)
    {
        try {
            $producto = Producto::findOrFail($id);
            $producto->delete();

            return redirect()->route('productos.index')->with('success', 'Producto eliminado correctamente.');
        } catch (ModelNotFoundException $e) {
            return redirect()->route('productos.index')->with('error', 'Producto no encontrado.');
        }
    }
    public function exportExcel()
    {
        $productos = Producto::all()->map(function ($producto) {
            return [
                'ID' => $producto->id,
                'Nombre' => $producto->nombre,
                'Descripción' => $producto->descripcion,
                'Precio' => $producto->precio,
                'Stock' => $producto->stock,
                'Categoría' => optional($producto->categoria)->nombre,
            ];
        });
    
        return (new FastExcel($productos))->download('productos.xlsx');
    }




    public function verPDF()
    {
        $productos = Producto::all();
        $pdf = Pdf::loadView('productos.pdf', compact('productos'));
        return $pdf->stream('productos.pdf'); // 👈 Esto lo muestra en el navegador
    }
    

    public function vender($productoId, $alquilerId)
    {
        try {
            $producto = Producto::findOrFail($productoId);
            $alquiler = Alquiler::findOrFail($alquilerId);

            if ($producto->stock <= 0) {
                return redirect()->route('productos.index')->with('error', 'No hay stock disponible para vender.');
            }

            // Crear venta
            Venta::create([
                'producto_id' => $producto->id,
                'alquiler_id' => $alquiler->id,
                'precio_total' => $alquiler->precio, // Asegurar que esta columna existe en la tabla
            ]);

            // Reducir stock de manera segura
            $producto->decrement('stock', 1);

            return redirect()->route('productos.index')->with('success', 'Venta realizada con éxito.');
        } catch (ModelNotFoundException $e) {
            return redirect()->route('productos.index')->with('error', 'Producto o alquiler no encontrado.');
        } catch (\Exception $e) {
            return redirect()->route('productos.index')->with('error', 'Error al realizar la venta.');
        }
    }

    public function generarTicket($id)
    {
        try {
            $venta = Venta::findOrFail($id);

            $pdf = Pdf::loadView('productos.ticket', compact('venta'))
                ->setPaper([0, 0, 226.500, 600], 'portrait');

            return $pdf->stream("ticket_venta_{$venta->id}.pdf");
        } catch (ModelNotFoundException $e) {
            return back()->with('error', 'No se encontró la venta.');
        }
    }

    public function listarProductos()
    {
        try {
            $productos = Producto::select('id', 'nombre', 'precio')->get();

            if ($productos->isEmpty()) {
                return response()->json(['mensaje' => 'No hay productos disponibles'], 404);
            }

            return response()->json($productos);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error al obtener productos', 'detalle' => $e->getMessage()], 500);
        }
    }

    public function obtenerProductosStockBajo()
{
    $umbral = 5;  // Umbral de stock bajo
    $productosStockBajo = Producto::where('stock', '<', $umbral)->get();

    // Devolver la respuesta en formato JSON
    return response()->json($productosStockBajo);
}


}
