<?php

namespace App\Http\Controllers;
use App\Models\VentaDetalle;
use Illuminate\Http\Request;
use App\Models\Alquiler;
use App\Models\Venta;
use Rap2hpoutre\FastExcel\FastExcel;

class ReporteController extends Controller
{
    public function index(Request $request)
    {
        if (!auth()->user()->can('crear peliculas')) {
            return view('errors.access-denied'); // O usa abort(403) si prefieres
        }
    
        $fecha = $request->get('fecha') ?? date('Y-m-d');
        $metodo_pago = $request->get('metodo_pago');
    
        $ventas = VentaDetalle::with('producto')
            ->whereDate('created_at', $fecha)
            ->get();
    
        $totalVendido = $ventas->sum('subtotal');
    
        $alquileresQuery = Alquiler::whereDate('created_at', $fecha)
                                   ->where('estado', 'pagado');
    
        if ($metodo_pago) {
            $alquileresQuery->where('metodo_pago', $metodo_pago);
        }
    
        $alquileres = $alquileresQuery->get();
        $totalAlquileres = $alquileres->sum('precio');
    
        return view('reporte.index', compact(
            'fecha',
            'ventas',
            'totalVendido',
            'alquileres',
            'totalAlquileres',
            'metodo_pago'
        ));
    }
    

public function exportVentasExcel(Request $request)
{
    $fechaSeleccionada = $request->input('fecha');

    // Usamos VentaDetalle y traemos el producto relacionado
    $ventas = VentaDetalle::with('producto')
        ->whereDate('created_at', $fechaSeleccionada)
        ->get();

    if ($ventas->isEmpty()) {
        return redirect()->back()->with('error', 'No se encontraron ventas para la fecha seleccionada.');
    }

    return (new FastExcel($ventas))->download("ventas_{$fechaSeleccionada}.xlsx", function ($venta) {
        return [
            'ID' => $venta->id,
            'Producto' => $venta->producto->nombre ?? 'Producto eliminado',
            'Cantidad' => $venta->cantidad,
            'Subtotal' => number_format($venta->subtotal, 2),
            'Hora' => $venta->created_at->format('H:i:s'),
        ];
    });
}

    // ✅ Exportación de ALQUILERES a Excel
    public function exportAlquileresExcel(Request $request)
    {
        $fechaSeleccionada = $request->input('fecha');
        $metodo = $request->input('metodo_pago');

        $query = Alquiler::whereDate('created_at', $fechaSeleccionada)->where('estado', 'pagado');

        if ($metodo) {
            $query->where('metodo_pago', $metodo);
        }

        $alquileres = $query->get();

        if ($alquileres->isEmpty()) {
            return redirect()->back()->with('error', 'No se encontraron alquileres para la fecha seleccionada.');
        }

        return (new FastExcel($alquileres))->download("alquileres_{$fechaSeleccionada}.xlsx", function ($alquiler) {
            return [
                'ID' => $alquiler->id,
                'Cliente' => $alquiler->cliente,
                'Horas' => $alquiler->horas ?? 'N/A',
                'Precio' => $alquiler->precio,
                'Método de Pago' => ucfirst($alquiler->metodo_pago ?? 'N/A'),
                'Hora' => $alquiler->created_at->format('H:i:s'),
            ];
        });
    }
}
