@extends('adminlte::page')

@section('title', 'Detalle del Producto')

@section('content_header')
    <div class="d-flex justify-content-between align-items-center">
        <h1 class="m-0 text-dark">
            <i class="fas fa-box-open text-primary mr-2"></i>Detalle del Producto
        </h1>
        <div>
            <a href="{{ route('productos.edit', $producto->id) }}" class="btn btn-warning btn-sm">
                <i class="fas fa-edit mr-1"></i> Editar
            </a>
            <a href="{{ route('productos.index') }}" class="btn btn-secondary btn-sm">
                <i class="fas fa-arrow-left mr-1"></i> Volver
            </a>
        </div>
    </div>
@stop

@section('content')
<div class="container-fluid">
    <div class="row">
        <!-- Columna izquierda - Imagen y acciones -->
        <div class="col-md-4">
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-primary text-white">
                    <h5 class="card-title mb-0">Imagen del Producto</h5>
                </div>
                <div class="card-body text-center p-4">
                    @if($producto->imagen)
                        <img src="{{ asset('storage/' . $producto->imagen) }}" 
                             class="img-fluid rounded shadow-lg mb-3" 
                             style="max-height: 300px; object-fit: contain;"
                             alt="{{ $producto->nombre }}">
                    @else
                        <div class="bg-light p-5 rounded">
                            <i class="fas fa-box-open fa-5x text-muted mb-3"></i>
                            <p class="text-muted">No hay imagen disponible</p>
                        </div>
                    @endif
                    
                    <div class="d-flex justify-content-center mt-3">
                        
                       
                        <button class="btn btn-outline-info btn-sm" data-toggle="modal" data-target="#qrModal">
                            <i class="fas fa-qrcode mr-1"></i> Ver QR
                        </button>
                    </div>
                </div>
            </div>
            
            <!-- Tarjeta de stock -->
            <div class="card shadow-sm">
                <div class="card-header bg-primary text-white">
                    <h5 class="card-title mb-0">Gestión de Stock</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <h6 class="font-weight-bold">Nivel de Stock</h6>
                        <div class="progress" style="height: 25px;">
                            @php
                                $porcentaje = ($producto->stock / ($producto->stock_maximo ?: 100)) * 100;
                                $color = $porcentaje < 20 ? 'bg-danger' : ($porcentaje < 50 ? 'bg-warning' : 'bg-success');
                            @endphp
                            <div class="progress-bar {{ $color }}" role="progressbar" 
                                 style="width: {{ $porcentaje }}%;" 
                                 aria-valuenow="{{ $porcentaje }}" 
                                 aria-valuemin="0" 
                                 aria-valuemax="100">
                                {{ $producto->stock }} unidades
                            </div>
                        </div>
                        <small class="text-muted">
                            @if($producto->stock_maximo)
                                Máximo: {{ $producto->stock_maximo }} unidades
                            @else
                                Stock máximo no definido
                            @endif
                        </small>
                    </div>
                    
                    <div class="d-flex justify-content-between">
                        <div class="text-center">
                            <h6 class="font-weight-bold">Última Compra</h6>
                            <p class="mb-0">
                                {{ $producto->ultima_compra ? $producto->ultima_compra->format('d/m/Y') : 'N/A' }}
                            </p>
                        </div>
                        <div class="text-center">
                            <h6 class="font-weight-bold">Última Venta</h6>
                            <p class="mb-0">
                                {{ $producto->ultima_venta ? $producto->ultima_venta->format('d/m/Y') : 'N/A' }}
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Columna derecha - Detalles -->
        <div class="col-md-8">
            <div class="card shadow-sm">
                <div class="card-header bg-primary text-white">
                    <h5 class="card-title mb-0">Información General</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-4">
                                <h6 class="font-weight-bold text-primary">Nombre del Producto</h6>
                                <p class="lead">{{ $producto->nombre }}</p>
                            </div>
                            
                            <div class="mb-4">
                                <h6 class="font-weight-bold text-primary">Código</h6>
                                <p>{{ $producto->codigo ?? 'N/A' }}</p>
                            </div>
                            
                            <div class="mb-4">
                                <h6 class="font-weight-bold text-primary">Precios</h6>
                                <div class="table-responsive">
                                    <table class="table table-sm table-bordered">
                                        <tr>
                                            <td><strong>Precio de Venta</strong></td>
                                            <td class="text-right">S/ {{ number_format($producto->precio, 2) }}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Precio de Compra</strong></td>
                                            <td class="text-right">S/ {{ number_format($producto->precio_compra ?? 0, 2) }}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Margen de Ganancia</strong></td>
                                            <td class="text-right">
                                                @if($producto->precio_compra)
                                                    {{ number_format((($producto->precio - $producto->precio_compra) / $producto->precio_compra) * 100, 2) }}%
                                                @else
                                                    N/A
                                                @endif
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-4">
                                <h6 class="font-weight-bold text-primary">Descripción</h6>
                                <p>{{ $producto->descripcion ?? 'No especificada' }}</p>
                            </div>
                            
                            <div class="mb-4">
                                <h6 class="font-weight-bold text-primary">Categoría</h6>
                                <span class="badge badge-primary p-2">
                                    {{ $producto->categoria->nombre ?? 'Sin categoría' }}
                                </span>
                            </div>
                            
                            <div class="mb-4">
                                <h6 class="font-weight-bold text-primary">Estado Actual</h6>
                                @if($producto->stock == 0)
                                    <span class="badge badge-danger p-2">
                                        <i class="fas fa-times-circle mr-1"></i> Agotado
                                    </span>
                                @elseif($producto->stock < 5)
                                    <span class="badge badge-warning p-2">
                                        <i class="fas fa-exclamation-triangle mr-1"></i> Stock Bajo
                                    </span>
                                @else
                                    <span class="badge badge-success p-2">
                                        <i class="fas fa-check-circle mr-1"></i> Disponible
                                    </span>
                                @endif
                            </div>
                            
                            <div class="mb-4">
                                <h6 class="font-weight-bold text-primary">Fecha de Registro</h6>
                                <p>{{ $producto->created_at->format('d/m/Y H:i') }}</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal QR -->
<div class="modal fade" id="qrModal" tabindex="-1" role="dialog" aria-labelledby="qrModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-sm" role="document">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title" id="qrModalLabel">Código QR del Producto</h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body text-center">
                <div id="qrcode" class="mb-3"></div>
                <p class="small text-muted">Escanea este código para acceder rápidamente a la información del producto</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cerrar</button>
                <button type="button" class="btn btn-primary" onclick="downloadQR()">
                    <i class="fas fa-download mr-1"></i> Descargar
                </button>
            </div>
        </div>
    </div>
</div>
@stop

@section('css')
<style>
    .card {
        border-radius: 10px;
        overflow: hidden;
    }
    .card-header {
        padding: 12px 20px;
    }
    .progress {
        border-radius: 20px;
    }
    .progress-bar {
        font-size: 0.85rem;
        display: flex;
        align-items: center;
        justify-content: center;
    }
    #qrcode {
        display: inline-block;
        padding: 10px;
        border: 1px solid #eee;
    }
    @media print {
        .no-print {
            display: none !important;
        }
        .card {
            border: none;
            box-shadow: none;
        }
    }
</style>
@stop

@section('js')
<script src="https://cdn.rawgit.com/davidshimjs/qrcodejs/gh-pages/qrcode.min.js"></script>
<script>
$(document).ready(function() {
    // Generar QR
    new QRCode(document.getElementById("qrcode"), {
        text: "{{ route('productos.show', $producto->id) }}",
        width: 200,
        height: 200,
        colorDark : "#000000",
        colorLight : "#ffffff",
        correctLevel : QRCode.CorrectLevel.H
    });
});

function downloadQR() {
    const canvas = document.querySelector("#qrcode canvas");
    const link = document.createElement('a');
    link.download = 'QR-{{ Str::slug($producto->nombre) }}.png';
    link.href = canvas.toDataURL('image/png');
    link.click();
}
</script>
@stop